<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\User;

use Erlage\Photogram\Tools\Crypto;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Data\Models\User\UserFinder;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Exceptions\SessionException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Models\User\UserBuilder;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;

final class UserAuth extends ExceptionalRequests
{
    /*
    |--------------------------------------------------------------------------
    | session request [auto]
    |--------------------------------------------------------------------------
    */

    public static function session(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | ensure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated(ResponseConstants::D_ERROR_SESSION_UNAUTHORIZED_MSG);

            /*
            |--------------------------------------------------------------------------
            | return authed user
            |--------------------------------------------------------------------------
            */

            self::addToResponse(UserTable::getTableName(), self::$authedUserModel -> getPrivateMap());
        });
    }

    /*
    |--------------------------------------------------------------------------
    | login request {username, password}
    |--------------------------------------------------------------------------
    */

    public static function login(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $usernameFromReq = self::$request -> findKey(UserTable::USERNAME, RequestTable::PAYLOAD, UserTable::TABLE_NAME);
            $passwordFromReq = self::$request -> findKey(UserTable::PASSWORD, RequestTable::PAYLOAD, UserTable::TABLE_NAME);

            self::ensureValue(ResponseConstants::D_ERROR_USER_MISSING_FIELDS_MSG, $usernameFromReq, $passwordFromReq);

            /*
            |--------------------------------------------------------------------------
            | check whether it's a email
            |--------------------------------------------------------------------------
            */

            $isEmail = \filter_var($usernameFromReq, FILTER_VALIDATE_EMAIL);

            /*
            |--------------------------------------------------------------------------
            | try fetching the user using the provided username/email
            |--------------------------------------------------------------------------
            */

            $finder = (new UserFinder());

            if ($isEmail)
            {
                $finder -> setEmail($usernameFromReq);
            }
            else
            {
                $finder -> setUsername($usernameFromReq);
            }

            $finder -> find();

            if ($finder -> notFound())
            {
                throw new RequestException(ResponseConstants::D_ERROR_USER_NOT_FOUND_MSG);
            }

            $userModel = $finder -> popModelFromResults();

            /*
            |--------------------------------------------------------------------------
            | verify the provided password
            |--------------------------------------------------------------------------
            */

            if ( ! Crypto::verifyClearPasswordWithDatabaseHash($passwordFromReq, $userModel -> getPassword()))
            {
                throw new RequestException(ResponseConstants::D_ERROR_USER_NOT_MATCHED_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | do server side login
            |--------------------------------------------------------------------------
            */

            self::$userSession -> loginAs($userModel);

            self::userAuthenticate();

            /*
            |--------------------------------------------------------------------------
            | ensuring call make sure that user is allowed to construct sessions
            | this will throw in case of ban, email not verified etc
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | add user to response
            |--------------------------------------------------------------------------
            */

            self::addToResponse(UserTable::getTableName(), $userModel -> getPrivateMap());
        });
    }

    /*
    |--------------------------------------------------------------------------
    | logout request
    |--------------------------------------------------------------------------
    */

    public static function logout(): void
    {
        self::process(function ()
        {
            self::$userSession -> logout();

            throw new SessionException(ResponseConstants::SUCCESS_MSG);
        });
    }

    /*
    |--------------------------------------------------------------------------
    | register request {email, username, password}
    |--------------------------------------------------------------------------
    */

    public static function register(): void
    {
        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $emailFromReq = self::$request -> findKey(UserTable::EMAIL, RequestTable::PAYLOAD, UserTable::TABLE_NAME);
            $usernameFromReq = self::$request -> findKey(UserTable::USERNAME, RequestTable::PAYLOAD, UserTable::TABLE_NAME);
            $passwordFromReq = self::$request -> findKey(UserTable::PASSWORD, RequestTable::PAYLOAD, UserTable::TABLE_NAME);

            self::ensureValue(ResponseConstants::D_ERROR_USER_MISSING_FIELDS_MSG, $emailFromReq, $usernameFromReq, $passwordFromReq);

            /*
            |--------------------------------------------------------------------------
            | try register new user
            |--------------------------------------------------------------------------
            */

            $newUserModel = (new UserBuilder())
                -> setEmail($emailFromReq)
                -> setUsername($usernameFromReq)
                -> setPassword($passwordFromReq)
                -> dispense();

            /*
            |--------------------------------------------------------------------------
            | save user
            |--------------------------------------------------------------------------
            */

            $newUserModel -> save();

            /*
            |--------------------------------------------------------------------------
            | do server side login
            |--------------------------------------------------------------------------
            */

            self::$userSession -> loginAs($newUserModel);

            self::userAuthenticate();

            /*
            |--------------------------------------------------------------------------
            | ensuring call make sure that user is allowed to construct sessions
            | this will throw in case of ban, email not verified etc
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | success reply the user object
            |--------------------------------------------------------------------------
            */

            self::addToResponse(UserTable::getTableName(), $newUserModel -> getPrivateMap());
        });
    }
}
